<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class Voucher extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'voucher_no',
        'type',
        'date',
        'reference',
        'payment_mode',
        'account_id',
        'total_amount',
        'narration',
        'status',
        'created_by',
    ];

    protected $casts = [
        'date' => 'date',
        'total_amount' => 'decimal:2',
    ];

    /**
     * Voucher type labels
     */
    public const TYPES = [
        'credit' => 'Credit Voucher (Receipt)',
        'debit' => 'Debit Voucher (Payment)',
        'journal' => 'Journal Voucher',
    ];

    /**
     * Cash/Bank account relationship
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /**
     * Created by user relationship
     */
    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Voucher entries relationship
     */
    public function entries(): HasMany
    {
        return $this->hasMany(VoucherEntry::class);
    }

    /**
     * Transactions relationship
     */
    public function transactions(): MorphMany
    {
        return $this->morphMany(Transaction::class, 'transactionable');
    }

    /**
     * Get voucher type label
     */
    public function getTypeLabelAttribute(): string
    {
        return self::TYPES[$this->type] ?? $this->type;
    }

    /**
     * Calculate total from entries
     */
    public function calculateTotal(): void
    {
        if ($this->type === 'journal') {
            $this->total_amount = $this->entries->sum('debit');
        } else {
            $this->total_amount = $this->entries->sum(function ($entry) {
                return $this->type === 'credit' ? $entry->credit : $entry->debit;
            });
        }
    }

    /**
     * Scope by type
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope for confirmed vouchers
     */
    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, $from, $to)
    {
        return $query->whereBetween('date', [$from, $to]);
    }

    /**
     * Generate next voucher number
     */
    public static function generateVoucherNo(string $type): string
    {
        $prefixes = [
            'credit' => 'CR-',
            'debit' => 'DR-',
            'journal' => 'JV-',
        ];

        $prefix = $prefixes[$type] ?? 'V-';
        $year = date('y');
        $month = date('m');
        
        $lastVoucher = self::where('voucher_no', 'like', $prefix . $year . $month . '%')
            ->orderBy('voucher_no', 'desc')
            ->first();

        if ($lastVoucher) {
            $lastNumber = intval(substr($lastVoucher->voucher_no, -4));
            return $prefix . $year . $month . str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        }

        return $prefix . $year . $month . '0001';
    }

    /**
     * Post to ledger (create transactions)
     */
    public function postToLedger(): void
    {
        // Delete existing transactions
        $this->transactions()->delete();

        if ($this->status !== 'confirmed') {
            return;
        }

        $voucherNo = $this->voucher_no;
        $date = $this->date;
        $userId = $this->created_by;
        $voucherType = strtoupper(substr($this->type, 0, 2)); // CR, DR, JV

        foreach ($this->entries as $entry) {
            $this->transactions()->create([
                'date' => $date,
                'account_id' => $entry->account_id,
                'party_id' => $entry->party_id,
                'voucher_type' => $voucherType,
                'voucher_no' => $voucherNo,
                'description' => $entry->description ?? $this->narration,
                'debit' => $entry->debit,
                'credit' => $entry->credit,
                'created_by' => $userId,
            ]);
        }

        // For Credit/Debit vouchers, also post to Cash/Bank account
        if ($this->type !== 'journal' && $this->account_id) {
            $this->transactions()->create([
                'date' => $date,
                'account_id' => $this->account_id,
                'party_id' => null,
                'voucher_type' => $voucherType,
                'voucher_no' => $voucherNo,
                'description' => $this->narration,
                'debit' => $this->type === 'credit' ? $this->total_amount : 0,
                'credit' => $this->type === 'debit' ? $this->total_amount : 0,
                'created_by' => $userId,
            ]);
        }

        // Update account balances
        $this->updateAccountBalances();
    }

    /**
     * Update all related account balances
     */
    protected function updateAccountBalances(): void
    {
        $accountIds = $this->transactions->pluck('account_id')->unique();
        
        foreach ($accountIds as $accountId) {
            $account = Account::find($accountId);
            if ($account) {
                $account->updateBalance();
            }
        }

        // Update party balances
        $partyIds = $this->entries->pluck('party_id')->filter()->unique();
        foreach ($partyIds as $partyId) {
            $party = Party::find($partyId);
            if ($party) {
                $party->updateBalance();
            }
        }
    }
}
