<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Item extends Model
{
    use HasFactory;

    /**
     * Product types
     */
    public const TYPES = [
        'raw' => 'Raw Material',
        'finish' => 'Finished Goods',
    ];

    /**
     * Units of measurement
     */
    public const UNITS = [
        'kg' => 'Kilogram (KG)',
        'g' => 'Gram (G)',
        'ton' => 'Metric Ton',
        'pcs' => 'Pieces (PCS)',
        'mtr' => 'Meter (MTR)',
        'ltr' => 'Liter (LTR)',
        'bag' => 'Bag',
        'bale' => 'Bale',
        'bundle' => 'Bundle',
        'roll' => 'Roll',
    ];

    protected $fillable = [
        'code',
        'name',
        'unit',
        'product_type',
        'category',
        'purchase_price',
        'sale_price',
        'opening_stock',
        'current_stock',
        'min_stock',
        'description',
        'is_active',
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'sale_price' => 'decimal:2',
        'opening_stock' => 'decimal:3',
        'current_stock' => 'decimal:3',
        'min_stock' => 'decimal:3',
        'is_active' => 'boolean',
    ];

    /**
     * Purchase items relationship
     */
    public function purchaseItems(): HasMany
    {
        return $this->hasMany(PurchaseItem::class);
    }

    /**
     * Sale items relationship
     */
    public function saleItems(): HasMany
    {
        return $this->hasMany(SaleItem::class);
    }

    /**
     * Check if stock is low
     */
    public function getIsLowStockAttribute(): bool
    {
        return $this->current_stock <= $this->min_stock;
    }

    /**
     * Check if out of stock
     */
    public function getIsOutOfStockAttribute(): bool
    {
        return $this->current_stock <= 0;
    }

    /**
     * Get stock value (current stock * purchase price)
     */
    public function getStockValueAttribute(): float
    {
        return $this->current_stock * $this->purchase_price;
    }

    /**
     * Update stock after purchase
     */
    public function addStock(float $quantity): void
    {
        $this->current_stock += $quantity;
        $this->save();
    }

    /**
     * Update stock after sale
     */
    public function reduceStock(float $quantity): void
    {
        $this->current_stock -= $quantity;
        $this->save();
    }

    /**
     * Scope for active items
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for RAW materials
     */
    public function scopeRaw($query)
    {
        return $query->where('product_type', 'raw');
    }

    /**
     * Scope for FINISH products
     */
    public function scopeFinish($query)
    {
        return $query->where('product_type', 'finish');
    }

    /**
     * Scope for low stock items
     */
    public function scopeLowStock($query)
    {
        return $query->whereColumn('current_stock', '<=', 'min_stock');
    }

    /**
     * Generate next item code
     */
    public static function generateCode(string $productType = 'raw'): string
    {
        $prefix = $productType === 'raw' ? 'R' : 'F';
        $lastItem = self::where('code', 'like', $prefix . '%')
            ->orderBy('code', 'desc')
            ->first();

        if ($lastItem) {
            $lastNumber = intval(substr($lastItem->code, 1));
            return $prefix . str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        }

        return $prefix . '0001';
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($item) {
            // Generate code if not provided
            if (empty($item->code)) {
                $item->code = self::generateCode($item->product_type);
            }

            // Set current stock to opening stock
            if ($item->current_stock == 0 && $item->opening_stock > 0) {
                $item->current_stock = $item->opening_stock;
            }
        });
    }
}
