<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Contract extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'contract_no',
        'title',
        'party_id',
        'type',
        'start_date',
        'end_date',
        'item_id',
        'agreed_rate',
        'agreed_quantity',
        'quantity_unit',
        'delivered_quantity',
        'pending_quantity',
        'agent_commission_rate',
        'transport_rate',
        'terms',
        'remarks',
        'status',
        'created_by',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'agreed_rate' => 'decimal:2',
        'agreed_quantity' => 'decimal:3',
        'delivered_quantity' => 'decimal:3',
        'pending_quantity' => 'decimal:3',
        'agent_commission_rate' => 'decimal:2',
        'transport_rate' => 'decimal:2',
    ];

    const TYPES = [
        'purchase' => 'Purchase Contract',
        'sale' => 'Sale Contract',
    ];

    const STATUSES = [
        'draft' => 'Draft',
        'active' => 'Active',
        'completed' => 'Completed',
        'cancelled' => 'Cancelled',
    ];

    const UNITS = [
        'kg' => 'Kilogram',
        'ton' => 'Ton',
        'maund' => 'Maund',
        'bale' => 'Bale',
        'pcs' => 'Pieces',
    ];

    // Relationships
    public function party()
    {
        return $this->belongsTo(Party::class);
    }

    public function item()
    {
        return $this->belongsTo(Item::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function purchases()
    {
        return $this->hasMany(Purchase::class);
    }

    public function sales()
    {
        return $this->hasMany(Sale::class);
    }

    // Generate unique contract number
    public static function generateContractNo($type = 'purchase')
    {
        $prefix = $type === 'purchase' ? 'PC-' : 'SC-';
        $year = date('Y');
        $lastContract = self::where('contract_no', 'like', $prefix . $year . '%')
            ->orderBy('contract_no', 'desc')
            ->first();

        if ($lastContract) {
            $lastNumber = intval(substr($lastContract->contract_no, -4));
            $newNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $newNumber = '0001';
        }

        return $prefix . $year . '-' . $newNumber;
    }

    // Calculate pending quantity
    public function calculatePendingQuantity()
    {
        return max(0, $this->agreed_quantity - $this->delivered_quantity);
    }

    // Update delivered quantity
    public function updateDeliveredQuantity($quantity, $action = 'add')
    {
        if ($action === 'add') {
            $this->delivered_quantity += $quantity;
        } else {
            $this->delivered_quantity -= $quantity;
        }
        
        $this->pending_quantity = $this->calculatePendingQuantity();
        
        // Auto-complete if fully delivered
        if ($this->pending_quantity <= 0 && $this->status === 'active') {
            $this->status = 'completed';
        }
        
        $this->save();
    }

    // Get contract value
    public function getContractValueAttribute()
    {
        return $this->agreed_rate * $this->agreed_quantity;
    }

    // Get delivered value
    public function getDeliveredValueAttribute()
    {
        return $this->agreed_rate * $this->delivered_quantity;
    }

    // Get pending value
    public function getPendingValueAttribute()
    {
        return $this->agreed_rate * $this->pending_quantity;
    }

    // Get completion percentage
    public function getCompletionPercentageAttribute()
    {
        if ($this->agreed_quantity <= 0) return 0;
        return round(($this->delivered_quantity / $this->agreed_quantity) * 100, 2);
    }

    // Check if contract is expired
    public function getIsExpiredAttribute()
    {
        return $this->end_date && $this->end_date < now();
    }

    // Check if contract is active
    public function getIsActiveAttribute()
    {
        return $this->status === 'active' && !$this->is_expired;
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopePurchase($query)
    {
        return $query->where('type', 'purchase');
    }

    public function scopeSale($query)
    {
        return $query->where('type', 'sale');
    }

    public function scopeByParty($query, $partyId)
    {
        return $query->where('party_id', $partyId);
    }

    public function scopePending($query)
    {
        return $query->where('pending_quantity', '>', 0);
    }
}
