<?php

namespace App\Http\Controllers;

use App\Models\Item;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;

class ItemController extends Controller
{
    /**
     * Display a listing of items.
     */
    public function index(Request $request): View
    {
        $query = Item::query();

        // Filter by type
        if ($request->filled('type')) {
            $query->where('product_type', $request->type);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('code', 'like', "%{$search}%")
                  ->orWhere('name', 'like', "%{$search}%");
            });
        }

        // Only active
        if ($request->boolean('active_only', true)) {
            $query->where('is_active', true);
        }

        $items = $query->orderBy('name')->get();

        $types = Item::TYPES;

        return view('items.index', compact('items', 'types'));
    }

    /**
     * Show the form for creating a new item.
     */
    public function create(): View
    {
        $types = Item::TYPES;
        $units = Item::UNITS;
        return view('items.create', compact('types', 'units'));
    }

    /**
     * Store a newly created item.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'type' => 'required|in:raw,finish',
            'unit' => 'required|string|max:20',
            'purchase_price' => 'nullable|numeric|min:0',
            'sale_price' => 'nullable|numeric|min:0',
            'opening_stock' => 'nullable|numeric|min:0',
            'minimum_stock' => 'nullable|numeric|min:0',
            'description' => 'nullable|string|max:255',
        ]);

        // Generate code
        $validated['code'] = Item::generateCode($validated['type']);
        $validated['opening_stock'] = $validated['opening_stock'] ?? 0;
        $validated['current_stock'] = $validated['opening_stock'];
        $validated['purchase_price'] = $validated['purchase_price'] ?? 0;
        $validated['sale_price'] = $validated['sale_price'] ?? 0;
        $validated['is_active'] = true;

        Item::create($validated);

        return redirect()->route('items.index')
            ->with('success', 'Item created successfully.');
    }

    /**
     * Display the specified item.
     */
    public function show(Item $item): View
    {
        $item->load(['purchaseItems.purchase', 'saleItems.sale']);
        
        return view('items.show', compact('item'));
    }

    /**
     * Show the form for editing the specified item.
     */
    public function edit(Item $item): View
    {
        $types = Item::TYPES;
        $units = Item::UNITS;
        return view('items.edit', compact('item', 'types', 'units'));
    }

    /**
     * Update the specified item.
     */
    public function update(Request $request, Item $item): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'type' => 'required|in:raw,finish',
            'unit' => 'required|string|max:20',
            'purchase_price' => 'nullable|numeric|min:0',
            'sale_price' => 'nullable|numeric|min:0',
            'opening_stock' => 'nullable|numeric|min:0',
            'minimum_stock' => 'nullable|numeric|min:0',
            'description' => 'nullable|string|max:255',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);

        // Recalculate current stock if opening stock changed
        if (isset($validated['opening_stock']) && $validated['opening_stock'] != $item->opening_stock) {
            $stockDiff = $validated['opening_stock'] - $item->opening_stock;
            $validated['current_stock'] = $item->current_stock + $stockDiff;
        }

        $item->update($validated);

        return redirect()->route('items.index')
            ->with('success', 'Item updated successfully.');
    }

    /**
     * Remove the specified item.
     */
    public function destroy(Item $item): RedirectResponse
    {
        // Check if item has transactions
        if ($item->purchaseItems()->exists() || $item->saleItems()->exists()) {
            return redirect()->route('items.index')
                ->with('error', 'Cannot delete item with transactions. Deactivate it instead.');
        }

        $item->delete();

        return redirect()->route('items.index')
            ->with('success', 'Item deleted successfully.');
    }

    /**
     * Get item details via AJAX
     */
    public function getDetails(Item $item)
    {
        return response()->json([
            'id' => $item->id,
            'code' => $item->code,
            'name' => $item->name,
            'type' => $item->type,
            'unit' => $item->unit,
            'purchase_price' => $item->purchase_price,
            'sale_price' => $item->sale_price,
            'current_stock' => $item->current_stock,
        ]);
    }
}
