<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Item;
use App\Models\Party;
use App\Models\Purchase;
use App\Models\Sale;
use App\Models\Setting;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\View\View;

class DashboardController extends Controller
{
    /**
     * Display the dashboard.
     */
    public function index(): View
    {
        // Get company information from settings
        $company = (object) [
            'name' => Setting::get('company_name', 'USMAN BILAL Enterprises'),
            'address' => Setting::get('company_address', 'Cotton Waste Processors & Dealers'),
            'phone' => Setting::get('company_phone', ''),
        ];

        // Financial year
        $financialYear = 'July ' . date('Y') . ' - June ' . (date('Y') + 1);

        // Today's purchase and sale
        $todayPurchase = Purchase::whereDate('date', today())->sum('total_amount');
        $todaySale = Sale::whereDate('date', today())->sum('total_amount');

        // Cash balance
        $cashAccount = Account::where('sub_type', 'Cash in Hand')->first();
        $cashBalance = 0;
        if ($cashAccount) {
            $cashBalance = Transaction::where('account_id', $cashAccount->id)
                ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                ->value('balance') ?? 0;
        }

        // Bank balance
        $bankAccounts = Account::where('sub_type', 'Bank Accounts')->pluck('id');
        $bankBalance = Transaction::whereIn('account_id', $bankAccounts)
            ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
            ->value('balance') ?? 0;

        // Total receivable (from customers)
        $customerAccounts = Party::customers()->with('account')->get()->pluck('account.id')->filter();
        $totalReceivable = 0;
        if ($customerAccounts->count() > 0) {
            $totalReceivable = Transaction::whereIn('account_id', $customerAccounts)
                ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                ->value('balance') ?? 0;
        }

        // Total payable (to suppliers)
        $supplierAccounts = Party::suppliers()->with('account')->get()->pluck('account.id')->filter();
        $totalPayable = 0;
        if ($supplierAccounts->count() > 0) {
            $totalPayable = abs(Transaction::whereIn('account_id', $supplierAccounts)
                ->selectRaw('COALESCE(SUM(debit), 0) - COALESCE(SUM(credit), 0) as balance')
                ->value('balance') ?? 0);
        }

        // Stock value
        $items = Item::all();
        $stockValue = $items->sum(fn($item) => $item->current_stock * $item->purchase_price);

        // Monthly profit (rough estimate: sales - purchases this month)
        $monthStart = now()->startOfMonth();
        $monthlySales = Sale::where('date', '>=', $monthStart)->sum('total_amount');
        $monthlyPurchases = Purchase::where('date', '>=', $monthStart)->sum('total_amount');
        $monthlyProfit = $monthlySales - $monthlyPurchases;

        // Recent transactions
        $recentTransactions = Transaction::with(['account', 'transactionable'])
            ->latest()
            ->take(10)
            ->get();

        // Low stock items (current_stock <= 10)
        $lowStockItems = Item::where('current_stock', '<=', 10)
            ->orderBy('current_stock')
            ->take(5)
            ->get();

        // Top customers by sales
        $topCustomers = Sale::with('party')
            ->selectRaw('party_id, SUM(total_amount) as total')
            ->groupBy('party_id')
            ->orderByDesc('total')
            ->take(5)
            ->get();

        return view('dashboard', compact(
            'company',
            'financialYear',
            'todayPurchase',
            'todaySale',
            'cashBalance',
            'bankBalance',
            'totalReceivable',
            'totalPayable',
            'stockValue',
            'monthlyProfit',
            'recentTransactions',
            'lowStockItems',
            'topCustomers'
        ));
    }
}
